#include <Arduino.h>
#include "LedMatrix.h"
#include "Snake.h"

Snake::Snake(LedMatrix* matrix)
{
    /* init matrix */
    this->matrix = matrix;

    /* init snake */
    for (int i = 0; i < SNAKE_MAX_SIZE; i++)
    {
        this->snake[i][0] = -1;
        this->snake[i][1] = -1;
    }
}

void Snake::initGame()
{
    /* init fields */
    this->moveLength_ms = 500;
    this->prevMove      = millis() / this->moveLength_ms;
    this->size = SNAKE_START_SIZE;
    /* init snake */
    for (int i = 0; i < this->size; i++)
    {
        if (this->direction == SNAKE_DIRECTION_LEFT)
        {
            this->snake[i][0] = 4 + i; this->snake[i][1] = 4;
        }
        else if (this->direction == SNAKE_DIRECTION_RIGHT)
        {
            this->snake[i][0] = 3 - i; this->snake[i][1] = 4;
        }
        else if (this->direction == SNAKE_DIRECTION_UP)
        {
            this->snake[i][0] = 3; this->snake[i][1] = 4 + i;
        }
        else if (this->direction == SNAKE_DIRECTION_DOWN)
        {
            this->snake[i][0] = 3; this->snake[i][1] = 3 - i;
        }
    }
    
    //Turn off the rest
    for (int i = this->size ; i < SNAKE_MAX_SIZE; i++)
    {
        this->snake[i][0] = -1; this->snake[i][1] = -1;
    }

    /* init food */
    this->newFood();

    /* init matrix */
    this->matrix->clear();
    for (int i = 0; i < this->size; i++) {
        this->matrix->turnOn(this->snake[i][0], this->snake[i][1], 1);
    }
}

void Snake::loop()
{
    /* set up matrix content if snake playing */
    if (this->gameStatus == SNAKE_STATUS_PLAYING)
    {
        /* update position */
        int currentMove = millis() / moveLength_ms;
        if (this->prevMove != currentMove)
        {
            this->moveSnake();
            this->prevMove = currentMove;
        }

        /* show food */
        this->matrix->turnOn(this->food[0], this->food[1], 1);

        /* ensure whole snake is turned on */
        for (int i = 0; i < this->size; i++) {
            this->matrix->turnOn(this->snake[i][0], this->snake[i][1], 1);
        }
    }

    /* update display */
    this->matrix->display();
}

void Snake::restart()
{
    this->matrix->clear();
    this->initGame();
    this->gameStatus = SNAKE_STATUS_PLAYING;
    this->nextDirection = this->direction;
}

void Snake::changeDirection(int direction)
{
    /* check this is a valid direction */
    if (direction == SNAKE_DIRECTION_LEFT || direction == SNAKE_DIRECTION_RIGHT
        || direction == SNAKE_DIRECTION_UP || direction == SNAKE_DIRECTION_DOWN)
    {
        /* if game is started, change the current direction of the snake */
        if (this->gameStatus == SNAKE_STATUS_PLAYING)
        {
            this->nextDirection = direction;
        }
        /* if game is not yet started, change the direction of the snake on start */
        else
        {
            this->direction = direction;
        }
    }
}

bool Snake::isInWall()
{
  if(this->gameStatus != SNAKE_STATUS_PLAYING)
  {
    return false;
  }
    for (int i = 1; this->isSnake(i); i++)
    {
        if (abs(this->snake[i-1][0] - this->snake[i][0]) > 1
            || abs(this->snake[i-1][1] - this->snake[i][1]) > 1)
        {
            return true;
        }
    }
    return false;
}

bool Snake::isInQueue()
{
  if(this->gameStatus != SNAKE_STATUS_PLAYING)
  {
    return false;
  }
    /* test if one part of the snake body is on another part of the snake body */
    for (int i = 0; i < this->size; i++)
    {
        int* currentPosition = this->snake[i];
        for (int j = i + 1; j < this->size; j++)
        { 
            if (currentPosition[0] == this->snake[j][0]
                && currentPosition[1] == this->snake[j][1])
            {
                return true;
            }
        }
    }
    return false;
}

bool Snake::isEating()
{
    return this->snake[0][0] == this->food[0]
        && this->snake[0][1] == this->food[1];
}

void Snake::grow()
{
    this->size++;
}

void Snake::newFood()
{
    this->matrix->set(this->food[0], this->food[1], LED_OFF);

    bool positionIsAlreadyUsedBySnake = true;
    while (positionIsAlreadyUsedBySnake)
    {
        this->food[0] = random(0, MATRIX_SIZE);
        this->food[1] = random(0, MATRIX_SIZE);

        positionIsAlreadyUsedBySnake = false;
        for (int i = 0; isSnake(i); i++)
        {
            if (this->snake[i][0] == this->food[0]
                && this->snake[i][1] == this->food[1])
            {
                positionIsAlreadyUsedBySnake = true;
                break;
            }
        }
    }
}

void Snake::accelerate()
{
    this->moveLength_ms -= this->moveLength_ms / 10;
    this->prevMove      = millis() / this->moveLength_ms;
}

void Snake::slowDown()
{
    this->moveLength_ms += this->moveLength_ms / 9;
    this->prevMove      = millis() / this->moveLength_ms;
}

int Snake::getSize()
{
    return this->size;
}

void Snake::loose()
{
    if (this->gameStatus == SNAKE_STATUS_PLAYING) {
        this->gameStatus = SNAKE_STATUS_LOOSE;
    }
}

void Snake::win()
{
    if (this->gameStatus == SNAKE_STATUS_PLAYING) {
        this->gameStatus = SNAKE_STATUS_WIN;
        //this->matrix->clear();
    }
}

bool Snake::isSnake(int position)
{
    return position >= 0 && position < SNAKE_MAX_SIZE
            && this->snake[position][0] >= 0 && this->snake[position][1] >= 0;
}

void Snake::moveSnake()
{
    bool isOppositeDirection =
         (this->direction == SNAKE_DIRECTION_LEFT && this->nextDirection == SNAKE_DIRECTION_RIGHT)
      || (this->direction == SNAKE_DIRECTION_RIGHT && this->nextDirection == SNAKE_DIRECTION_LEFT)
      || (this->direction == SNAKE_DIRECTION_UP && this->nextDirection == SNAKE_DIRECTION_DOWN)
      || (this->direction == SNAKE_DIRECTION_DOWN && this->nextDirection == SNAKE_DIRECTION_UP);

    /* if the new direction is the opposite direction, it is not updated as
    the snake will instantly be in its queue */
    if (!isOppositeDirection && this->direction != this->nextDirection)
    {
        this->direction = this->nextDirection;
    }
            
    //remove queue
    this->matrix->turnOff(this->snake[size - 1][0], this->snake[size - 1][1]);
    
    //shift
    for(int i=this->size-1; i>0; i--)
    {
      this->snake[i][0] = this->snake[i - 1][0];
      this->snake[i][1] = this->snake[i - 1][1];
    }
    
    // move the head
    switch (this->direction)
    {
        case SNAKE_DIRECTION_RIGHT:
            this->snake[0][0] = (this->snake[0][0] + 1) % MATRIX_SIZE;
            break;
        case SNAKE_DIRECTION_LEFT:
            this->snake[0][0]--;
            if (this->snake[0][0] < 0) { this->snake[0][0] = MATRIX_SIZE - 1; }
            break;
        case SNAKE_DIRECTION_UP:
            this->snake[0][1]--;
            if (this->snake[0][1] < 0) { this->snake[0][1] = MATRIX_SIZE - 1; }
            break;
        case SNAKE_DIRECTION_DOWN:
            this->snake[0][1] = (this->snake[0][1] + 1) % MATRIX_SIZE;
            break;
    } 
}